/*
 * $Id: $
 *
 * NetBSD ppbus/gpio Driver
 * Copyright (C) 2011 Frank Wille
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 * Written by Frank Wille <frank@phoenix.owl.de>, 2011.
 *
 */

#include <sysdep.h>

#include <fcntl.h>
#include <unistd.h>
#include <sys/gpio.h>
#include <sys/ioctl.h>

#include <stdlib.h>
#include <string.h>

#include <urjtag/error.h>
#include <urjtag/log.h>
#include <urjtag/parport.h>
#include <urjtag/cable.h>
#include "../parport.h"

static port_node_t *ports = NULL;       /* ppi parallel ports */

typedef struct {
	char *portname;
	int fd;
	uint8_t data, stat, ctrl;
} gpio_params_t;

static urj_parport_t *
gpio_parport_alloc (const char *port)
{
    gpio_params_t *params = malloc (sizeof *params);
    char *portname = strdup (port);
    urj_parport_t *parport = malloc (sizeof *parport);
    port_node_t *node = malloc (sizeof *node);

    if (!node || !parport || !params || !portname) {
        free (node);
        free (parport);
        free (params);
        free (portname);
        urj_error_set (URJ_ERROR_OUT_OF_MEMORY,
                       "malloc(%zd)/strdup(%s)/malloc(%zd)/malloc(%zd) fails",
                       sizeof *params, port, sizeof *parport, sizeof *node);
        return NULL;
    }

    params->portname = portname;
    params->fd = -1;

    parport->params = params;
    parport->driver = &urj_tap_parport_gpio_parport_driver;
    parport->cable = NULL;

    node->port = parport;
    node->next = ports;

    ports = node;

    return parport;
}

static void
gpio_parport_free (urj_parport_t *port)
{
    port_node_t **prev;

    for (prev = &ports; *prev; prev = &((*prev)->next))
        if ((*prev)->port == port)
            break;

    if (*prev) {
        port_node_t *pn = *prev;
        *prev = pn->next;
        free (pn);
    }

    free (((gpio_params_t *) port->params)->portname);
    free (port->params);
    free (port);
}

static urj_parport_t *
gpio_connect (const char *devname)
{
    port_node_t *pn;
    urj_parport_t *parport;

    for (pn = ports; pn; pn = pn->next)
        if (strcmp (pn->port->params, devname) == 0)
        {
            urj_log (URJ_LOG_LEVEL_NORMAL,
                     _("Disconnecting %s from gpio port %s\n"),
                     _(pn->port->cable->driver->description), devname);
            pn->port->cable->driver->disconnect (pn->port->cable);
            break;
        }

    urj_log (URJ_LOG_LEVEL_NORMAL, _("Initializing gpio port %s\n"), devname);

    parport = gpio_parport_alloc (devname);
    if (!parport)
        return NULL;

    return parport;
}

static int
gpio_open (urj_parport_t *parport)
{
    gpio_params_t *p = parport->params;
    struct gpio_req req;
    int pin;

    p->fd = open (p->portname, O_RDWR);
    if (p->fd < 0) {
        urj_error_IO_set ("Cannot open(%s)", p->portname);
        return URJ_STATUS_FAIL;
    }

    /* Initialize all GPIO pins with 0 */
    memset(&req, 0, sizeof(req));
    req.gp_value = GPIO_PIN_LOW;
    for (pin = 0; pin < 17; pin++) {
        req.gp_pin = pin;
        if (ioctl(p->fd, GPIOWRITE, &req) == -1) {
            close(p->fd);
            p->fd = -1;
            return URJ_STATUS_FAIL;
        }
    }
    p->data = p->stat = p->ctrl = 0;

    return URJ_STATUS_OK;
}

static int
gpio_close (urj_parport_t *parport)
{
    int r = URJ_STATUS_OK;
    gpio_params_t *p = parport->params;

    if (close (p->fd) != 0) {
        urj_error_IO_set ("Cannot close(%d)", p->fd);
        return URJ_STATUS_FAIL;
    }

    p->fd = -1;
    return r;
}

static int
gpio_set_data (urj_parport_t *parport, unsigned char data)
{
    gpio_params_t *p = parport->params;
    struct gpio_req req;
    uint8_t changed, mask;
    int i;

    memset(&req, 0, sizeof(req));

    if ((changed = data ^ p->data)) {
        for (mask = 1, i = 1; i <= 8; mask <<= 1, i++) {
            if ((changed & mask) != 0) {
                req.gp_pin = i;
                req.gp_value = (data & mask) ? GPIO_PIN_HIGH : GPIO_PIN_LOW;
                if (ioctl(p->fd, GPIOWRITE, &req) == -1)
                    return URJ_STATUS_FAIL;
            }
        }
        p->data = data;
    }

    return URJ_STATUS_OK;
}

static int
gpio_get_data (urj_parport_t *parport)
{
    gpio_params_t *p = parport->params;
    struct gpio_req req;
    int d, i, mask;

    memset(&req, 0, sizeof(req));

    for (d = 0, mask = 1, i = 1; i <= 8; mask <<= 1, i++) {
        req.gp_pin = i;
        if (ioctl( p->fd, GPIOREAD, &req ) == -1)
            return -1;
        if (req.gp_value == GPIO_PIN_HIGH)
            d |= mask;
    }
    p->data = d;

    return d;
}

static int
gpio_get_status (urj_parport_t *parport)
{
    gpio_params_t *p = parport->params;
    struct gpio_req req;
    int d, i, mask;
    int pins[8] = { 0, 0, 0, 14, 12, 11, 9, 10 };

    memset(&req, 0, sizeof(req));

    for (d = 0, mask = (1<<3), i = 3; i < 8; mask <<= 1, i++) {
        req.gp_pin = pins[i];
        if (ioctl( p->fd, GPIOREAD, &req ) == -1)
            return -1;
        if (req.gp_value == GPIO_PIN_HIGH)
            d |= mask;
    }
    p->stat = d;

    return d;
}

static int
gpio_set_control (urj_parport_t *parport, unsigned char data)
{
    gpio_params_t *p = parport->params;
    struct gpio_req req;
    unsigned char changed, mask;
    int i;
    int pins[4] = { 0, 13, 15, 16 };

    memset(&req, 0, sizeof(req));

    if ((changed = data ^ p->ctrl)) {
        for (mask = 1, i = 0; i < 4; mask <<= 1, i++) {
            if ((changed & mask) != 0) {
                req.gp_pin = pins[i];
                req.gp_value = (data & mask) ? GPIO_PIN_HIGH : GPIO_PIN_LOW;
                if (ioctl( p->fd, GPIOWRITE, &req ) == -1)
                    return URJ_STATUS_FAIL;
            }
        }
        p->ctrl = data;
    }

    return URJ_STATUS_OK;
}

const urj_parport_driver_t urj_tap_parport_gpio_parport_driver = {
    URJ_CABLE_PARPORT_DEV_GPIO,
    gpio_connect,
    gpio_parport_free,
    gpio_open,
    gpio_close,
    gpio_set_data,
    gpio_get_data,
    gpio_get_status,
    gpio_set_control
};
